/**
 * OWASP AppSensor
 * 
 * This file is part of the Open Web Application Security Project (OWASP)
 * AppSensor project. For details, please see
 * <a href="http://www.owasp.org/index.php/Category:OWASP_AppSensor_Project">
 * 	http://www.owasp.org/index.php/Category:OWASP_AppSensor_Project</a>.
 *
 * Copyright (c) 2010 - The OWASP Foundation
 * 
 * AppSensor is published by OWASP under the BSD license. You should read and accept the
 * LICENSE before you use, modify, and/or redistribute this software.
 * 
 * @author Michael Coates <a href="http://www.aspectsecurity.com">Aspect Security</a>
 * @author John Melton <a href="http://www.jtmelton.com/">jtmelton</a>
 * @created 2010
 */
namespace org.owasp.appsensor
{
    using System;
    using System.Collections.Generic;
    using System.Net;
    using System.Text.RegularExpressions;
    using System.Web;
    using org.owasp.appsensor.errors;
    using AppSensor2.configuration;

    /**
     * This class Contains a collection of methods for performing 
     * attack detection.  It is intended to house common methods that an 
     * application is likely to find useful when looking for suspicious 
     * behavior.  Typically these methods take necessary input, 
     * determine if an attack has occurred, then throw the appropriate
     * AppSensor related exception.  
     * 
     * @author Michael Coates (michael.coates .at. owasp.org) 
     *         <a href="http://www.aspectsecurity.com">Aspect Security</a>
     * @author John Melton (jtmelton .at. gmail.com) 
     *         <a href="http://www.jtmelton.com/">jtmelton</a>
     * @since February 24, 2010
     */
    public class AttackDetectorUtils
    {

        /** constant for HTTP GET */
        public const String GET = "GET";

        /** constant for HTTP POST */
        public const String POST = "POST";

        /** constant for null byte */
        public const String ENCODED_NULL_BYTE = "%00";

        /** constant for carriage return */
        public const String ENCODED_CARRIAGE_RETURN = "%0D";

        /** constant for line feed */
        public const String ENCODED_LINE_FEED = "%0A";

        /** Implementation of security config that pulls appropriate fields from appsensor.properties */
        private static AppSensorSecurityConfiguration assc = (AppSensorSecurityConfiguration)AppSensorSecurityConfiguration.GetInstance();

        /**
         * This method inspects the current HTTP request as well as the expected
         * HTTP request method, and determines if the request is performed with
         * an unexpected method, such as HEAD for instance.  If an unexpected 
         * method is found, the appropriate exception is instantiated (not thrown).
         * <p>
         * Note: This method only acts if the expected method passed in 
         * is either "GET" or "POST" (case-insensitive and ignoring the quotes)   
         * 
         * @see http://www.owasp.org/index.php/AppSensor_DetectionPoints#RE1:_Unexpected_HTTP_Commands
         * @see http://www.owasp.org/index.php/AppSensor_DetectionPoints#RE2:_Attempts_To_Invoke_Unsupported_HTTP_Methods
         * @see http://www.owasp.org/index.php/AppSensor_DetectionPoints#RE3:_GET_When_Expecting_POST
         * @see http://www.owasp.org/index.php/AppSensor_DetectionPoints#RE4:_POST_When_Expecting_GET
         * 
         * @param request the current HTTP request
         * @param expectedMethod the expected HTTP method for the current request
         * 
         * @return bool - true if request method is valid - false if not
         */
        public static bool verifyValidRequestMethod(HttpRequest request, String expectedMethod)
        {
            if (request == null)
            {
                throw new ArgumentException("HTTP Request cannot be null");
            }

            var allHttpMethods = assc.AllHttpMethods.ToList();
            var validHttpMethods = assc.ValidHttpMethods.ToList();

            bool isValid = true;

            if (expectedMethod != null && (expectedMethod.Equals(GET, StringComparison.OrdinalIgnoreCase)
                || expectedMethod.Equals(POST, StringComparison.OrdinalIgnoreCase)))
            {

                if (!validHttpMethods.Contains(request.RequestType))
                {
                    new AppSensorException("RE1", "AppSensorUser Message RE1", "Attacker is sending an invalid (valid, but not supported) command (" + request.RequestType + ") to the application");
                    isValid = false;
                }

                if (!allHttpMethods.Contains(request.RequestType))
                {
                    new AppSensorException("RE2", "AppSensorUser Message RE2", "Attacker is sending an invalid (invalid, does not exist) method (" + request.RequestType + ") to the application");
                    isValid = false;
                }

                if (!(request.RequestType.Equals(expectedMethod,StringComparison.OrdinalIgnoreCase)))
                {
                    if (expectedMethod.Equals(POST, StringComparison.OrdinalIgnoreCase))
                    {
                        // @AppSensor Attack Detection - RE3
                        new AppSensorException("RE3", "AppSensorUser Message RE3", "Attacker is sending a non-POST request (" + request.RequestType + ") to page designed for only POST");
                    }
                    else if (expectedMethod.Equals(GET, StringComparison.OrdinalIgnoreCase))
                    {
                        // @AppSensor Attack Detection - RE4
                        new AppSensorException("RE4", "AppSensorUser Message RE4", "Attacker is sending a non-GET request (" + request.RequestType + ") to page designed for only GET");
                    }
                    isValid = false;	//invalid request method
                }
                return isValid;
            }
            else
            {
                throw new ArgumentException("Method only allows GET and POST as valid expected methods.");
            }
        }

        /**
         * This method tests to see if the string passed in appears to be an explicit 
         * Cross Site Scripting (XSS) attack.  If it appears there is an attack, 
         * an appropriate exception is instantiated (not thrown).
         * <p>
         * Note: This method does not prevent XSS in an application or do filtering of any kind,
         * however it will fire an exception to the intrusion detector if an obvious attack exists.
         * 
         * @see http://www.owasp.org/index.php/AppSensor_DetectionPoints#IE1:_Cross_Site_Scripting_Attempt
         * 
         * @param testValue the string to test for an XSS attack, likely user input from a UI screen.
         * 
         * @return bool - true if XSS attack found, false if not
         */
        public static bool verifyXSSAttack(String testValue)
        {
            //AppSensor Checks on Search Value
            ICollection<String> attackPatterns = assc.XssAttackPatterns.ToList();

            bool attackFound = false;

            foreach (String pattern in attackPatterns)
            {
                if (testValue != null)
                {
                    bool regexFound = regexFind(testValue, pattern);
                    if (regexFound)
                    {
                        new AppSensorException("IE1", "AppSensorUser Message IE1", "Attacker is sending a likely XSS attempt (" + testValue + ").");
                        attackFound = true;
                    }
                }
            }

            return attackFound;
        }

        /**
         * This method tests to see if the string passed in appears to be an explicit 
         * SQL Injection attack.  If it appears there is an attack, 
         * an appropriate exception is instantiated (not thrown).
         * <p>
         * Note: This method does not prevent SQL Injection in an application or do filtering of any kind,
         * however it will fire an exception to the intrusion detector if an obvious attack exists.
         * 
         * @see http://www.owasp.org/index.php/AppSensor_DetectionPoints#CIE1:_Blacklist_Inspection_For_Common_SQL_Injection_Values
         * 
         * @param testValue the string to test for a SQL Injection attack, likely user input from a UI screen.
         * 
         * @return bool - true if SQL Injection attack found, false if not
         */
        public static bool verifySQLInjectionAttack(String testValue)
        {
            //AppSensor Checks on Search Value
            var attackPatterns = assc.SqlInjectionAttackPatterns.ToList();
            bool attackFound = false;

            foreach (String pattern in attackPatterns)
            {
                if (testValue != null)
                {
                    bool regexFound = regexFind(testValue, pattern);
                    if (regexFound)
                    {
                        new AppSensorException("CIE1", "AppSensorUser Message CIE1", "Attacker is sending a likely SQL Injection attempt (" + testValue + ").");
                        attackFound = true;
                    }
                }
            }

            return attackFound;
        }

        /**
         * This method inspects the current HTTP request as well as the expected
         * cookie names method, and determines if there are any differences, either
         * more or fewer cookies.  If a delta is detected, the appropriate exception 
         * is instantiated (not thrown).
         * <p>
         *  
         * @see http://www.owasp.org/index.php/AppSensor_DetectionPoints#SE2:_Adding_New_Cookies
         * @see http://www.owasp.org/index.php/AppSensor_DetectionPoints#SE3:_Deleting_Existing_Cookies
         * 
         * @param request the current HTTP request
         * @param expectedCookieNames the names of the expected cookies
         * 
         * @return bool - true if the expected cookie names are the only cookies received - false if not
         */
        public static bool verifyCookiesPresent(HttpRequest request, ICollection<String> expectedCookieNames)
        {
            if (request == null)
            {
                throw new ArgumentException("HTTP Request cannot be null");
            }
            if (expectedCookieNames == null)
            {
                throw new ArgumentException("Expected cookie names cannot be null");
            }

            bool onlyValidCookiesPresent = true;

            var cookies = request.Cookies;
            //check for null set of cookies and handle by creating empty array - same result
            if (cookies == null)
            {
                cookies = new HttpCookieCollection();
            }

            ICollection<String> cookieList = new List<String>();

            //get names of actual cookies that are in the request
            foreach (Cookie cookie in cookies)
            {
                cookieList.Add(cookie.Name);
            }

            //first make sure all expected cookie names are present in cookies list
            foreach (String ecName in expectedCookieNames)
            {
                if (cookieList.Contains(ecName))
                {
                    cookieList.Remove(ecName);	//remove so we can check leftovers below
                }
                else
                {
                    new AppSensorException("SE3", "AppSensorUser Message SE3", "Attacker has deleted an expected cookie named: " + ecName);
                    onlyValidCookiesPresent = false;
                }
            }

            //go through remaining cookies - if any exist, they must be invalid - added since we've already 
            //removed those that were expected
            foreach (String cookieName in cookieList)
            {
                new AppSensorException("SE4", "AppSensorUser Message SE4", "Attacker has added an un-expected cookie named: " + cookieName);
                onlyValidCookiesPresent = false;
            }

            return onlyValidCookiesPresent;
        }

        /**
         * This method checks a string to determine if a null byte exists in the string. 
         * Returns true if the string is clean (no null bytes)
         * <p>
         * 
         * @see http://www.owasp.org/index.php/AppSensor_DetectionPoints#CIE3:_Null_Byte_Character_In_File_Request
         * 
         * @param testString string to check for null bytes
         * 
         * @return true if the string is clean (no null bytes)
         */
        public static bool verifyNullByteDoesNotExist(String testString)
        {
            if (testString == null || testString.Trim().Equals(""))
            {
                return true;
            }
            else
            {
                bool containsNB = testString.Contains(ENCODED_NULL_BYTE);
                if (containsNB)
                {
                    new AppSensorException("CIE3", "AppSensorUser Message CIE3", "String Contains a Null Byte: " + testString);
                }
                return !containsNB;	//return opposite - ie. return true if null byte does not exist.
            }
        }

        /**
         * This method checks a string to determine if a carriage return or line feed exists in the string. 
         * Returns true only if the string Contains neither a carriage return or line feed, false if either or both exists
         * <p>
         * 
         * @see http://www.owasp.org/index.php/AppSensor_DetectionPoints#CIE4:_Carriage_Return_Or_Line_Feed_Character_In_File_Request
         * 
         * @param testString string to check for carriage return or line feed
         * 
         * @return true only if the string Contains neither a carriage return or line feed, false if either or both exists
         */
        public static bool verifyCarriageReturnOrLineFeedDoesNotExist(String testString)
        {
            if (testString == null || testString.Trim().Equals(""))
            {
                return true;
            }
            else
            {
                bool containsCR = testString.Contains(ENCODED_CARRIAGE_RETURN);
                bool containsLF = testString.Contains(ENCODED_LINE_FEED);
                if (containsCR && containsLF)
                {
                    new AppSensorException("CIE4", "AppSensorUser Message CIE4", "String Contains both a carriage return and line feed: " + testString);
                    return false;
                }
                else if (containsCR)
                {
                    new AppSensorException("CIE4", "AppSensorUser Message CIE4", "String Contains a carriage return: " + testString);
                    return false;
                }
                else if (containsLF)
                {
                    new AppSensorException("CIE4", "AppSensorUser Message CIE4", "String Contains a line feed: " + testString);
                    return false;
                }
                else
                {
                    //string is clean
                    return true;
                }
            }
        }

        /**
         * This method searches the target string to see if it matches the regex.
         * If so, it returns true, else false.
         * 
         * @param targetString string to test
         * @param regEx regular expression to search for
         * @return bool indicating whether test string matches regular expression or not
         */
        private static bool regexFind(String targetString, String regEx)
        {
            if (targetString == null)
            {
                return false;
            }

            var pattern = new Regex(regEx, RegexOptions.Compiled | RegexOptions.IgnoreCase);
            return pattern.IsMatch(targetString);
        }


    }
}